/**********************************************************************************
*   this file is part of c2h
*   Copyright (C)2005 Bruce Park ( jongsuknim@naver.com )
*
*   This program is free software; you can redistribute it and/or
*   modify it under the terms of the GNU General Public License
*   as published by the Free Software Foundation; either
*   version 2 of the License, or (at your option) any later version.
*
*   This program is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU General Public License for more details.
*
*   You should have received a copy of the GNU General Public License
*   along with this program; if not, write to the Free Software
*   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
***********************************************************************************/
#include "general.h"	

#include <string.h>
#include <limits.h>

#include "debug.h"
#include "keyword.h"
#include "routines.h"
#include "pltable.h"

#define HASH_EXPONENT	11		/* must be less than 17 */
#define NotFound    0
#define Found	    1
#define Itself	    2

/*
<{=--
    <img src="binary/hashentry.bmp">
--=}>
 */

typedef struct sHashEntry 
{
    lang_type	language;
    struct sEntryContent    *globalHead;
    struct sEntryContent    *staticHead;
    struct sHashEntry	*next;
    const char *keyword;
} hashEntry;


static const unsigned int TableSize = 1 << HASH_EXPONENT;
static hashEntry **HashTable = NULL;

/*
*   FUNCTION DEFINITIONS
*/

static hashEntry **getHashTable (void)
{
    static boolean allocated = FALSE;

    if (! allocated)
    {
	unsigned int i;

	HashTable = xMalloc (TableSize, hashEntry*);

	for (i = 0  ;  i < TableSize  ;  ++i)
	    HashTable [i] = NULL;

	allocated = TRUE;
    }
    return HashTable;
}

static hashEntry *getHashTableEntry (unsigned long hashedValue)
{
    hashEntry **const table = getHashTable ();
    hashEntry *entry;

    Assert (hashedValue < TableSize);
    entry = table [hashedValue];

    return entry;
}

static unsigned long hashValue (const char *const keyword)
{
    unsigned long value = 0;
    const unsigned char *p;

    Assert (keyword != NULL);

    /*  We combine the various words of the multiword key using the method
     *  described on page 512 of Vol. 3 of "The Art of Computer Programming".
     */
    for (p = (const unsigned char *) keyword  ;  *p != '\0'  ;  ++p)
    {
	value <<= 1;
	if (value & 0x00000100L)
	    value = (value & 0x000000ffL) + 1L;
	value ^= *p;
    }
    /*  Algorithm from page 509 of Vol. 3 of "The Art of Computer Programming"
     *  Treats "value" as a 16-bit integer plus 16-bit fraction.
     */
    value *= 40503L;		/* = 2^16 * 0.6180339887 ("golden ratio") */
    value &= 0x0000ffffL;	/* keep fractional part */
    value >>= 16 - HASH_EXPONENT; /* scale up by hash size and move down */

    return value;
}

static hashEntry *newEntry (const char *const keyword, lang_type language )
{
    hashEntry *const entry = xMalloc (1, hashEntry);

    entry->keyword  = keyword;
    entry->language = language;
    entry->globalHead	= NULL;
    entry->staticHead	= NULL;
    entry->next	= NULL;

    return entry;
}

static entryContent *newContent( const char *const source, const char *const struct_name, 
	char type, int line, int value)
{
    entryContent    *content = xMalloc(1, entryContent);

    content->line   = line;
    content->type   = type;
    content->value  = value;
    content->source = source;
    content->struct_name = struct_name;
    content->next   = NULL;
    content->itsDir = absoluteDirname(source);

    return content;
}

static void addContent	(hashEntry *entry, const char *const source, const char *const struct_name,
			char type, int line, scope s, int value)
{
    entryContent *head=NULL; 
    if( s == Global){
	head = entry->globalHead;
	if( head == NULL ){
	    entry->globalHead = newContent( source, struct_name, type , line, value );
	    return;
	}
    }
#ifdef DEBUG
    else if( s == Static ){
	head = entry->staticHead;
	if( head == NULL ){
	    entry->staticHead = newContent( source, struct_name, type, line, value );
	    return;
	}
    }
    else
	Assert( FALSE );
#else
    else{
	head = entry->staticHead;
	if( head == NULL ){
	    entry->staticHead = newContent( source, struct_name, type,  line , value);
	    return;
	}
    }
#endif
    

    entryContent *tmp,*prev=NULL, *new=NULL;
    for( tmp = head; tmp != NULL ; tmp = tmp->next )
    {
#if 0
	if( strcmp( tmp->source, source) == 0 && 
		tmp->line && line )
	{
	    return;
	}
#endif
	prev = tmp;
    }
    
    new	= newContent(  source, struct_name, type, line, value);

    if( prev != NULL ) {
	prev->next = new;
    } else 
	head->next = new;
}

extern void addKeyword (const char *const keyword, const char *const source, const char *const struct_name,
			lang_type language,char type, int line, scope s, int value)
{
    const unsigned long hashedValue = hashValue (keyword);
    hashEntry *tableEntry = getHashTableEntry (hashedValue);
    hashEntry *entry = tableEntry;

    if (entry == NULL) {
	hashEntry **const table = getHashTable ();
	table [hashedValue]	= newEntry (keyword, language );
	addContent( getHashTableEntry(hashedValue) , source, struct_name, type, line, s, value);
    } else {
	hashEntry *prev = NULL;
//	hashEntry *new	= NULL;

	while (entry != NULL)
	{
	    if (language == entry->language  &&
		strcmp (keyword, entry->keyword) == 0)
	    {
		if( s == Global )
		    addContent( entry, source, struct_name, type, line, s, value);
		else
		    addContent( entry, source, struct_name, type, line, s, value);
		break;
	    }
	    prev = entry;
	    entry = entry->next;
	}

	if (entry == NULL) {
	    Assert (prev != NULL);
	    prev->next = newEntry (keyword, language);
	    addContent( prev->next, source, struct_name, type, line, s , value);
	}
    }
    return;
}

static boolean is_match( char c, const char *string)
{
    int i;
    for( i = 0 ; string[i] != 0; i++ ){
	if( c == string[i] )
	    return TRUE;
    }
    return FALSE;
}

entryContent itself;

static int getStaticContent	(const hashEntry *const entry,const char *const keyword, int line, 
	const char *const source, const entryContent * *content, const char *element_type, boolean in_element)
{
    entryContent *tmp;
    int retval=NotFound;
    for( tmp = entry->staticHead; tmp != NULL ; tmp = tmp->next ){
	if( strcmp( tmp->source, source ) == 0 ){
	    if( tmp->line == line ){
		*content = tmp;
		return Itself;
	    }
	    else if( retval == NotFound ){
		/*
		    in_element̸, match̾ ã ̰
		    in_element ƴϸ, match ƴϾ ã ̴.
		 */
		if( in_element ) {
		    if(element_type != NULL && is_match( tmp->type, element_type )){
			retval = Found;
			*content = tmp;
		    }
		}
		if( !in_element ){ 
		    if(!(element_type != NULL && is_match( tmp->type, element_type ))){
			retval = Found;
			*content = tmp;
		    }
		}
	    }
	}
    }
    return retval;
}

/*
   global  켱 ϴµ
    source  丮 tmp 丮 ٸ װ ϵȴ. 
TODO:
   ׷ ʴٸ source 丮   κк ϵȴ.
    ̷ Ǿ ִµ, ̰  ϱ ؼ c include ؼ,
   java import ؼ ϴ° ʿ, Ƿ, 
   ׳ ó ִ 丮  ã  ʿϴ.
 */
static int getGlobalContent	(const hashEntry *const entry, const char *const keyword, int line, 
	const char *const source, const entryContent * *content, const char *element_type,  boolean in_element)
{
    int nowNum, maxNum=0;
    char *common_dir;
    entryContent *tmp;
    char *sourceDir = absoluteDirname(source);
    int retval=NotFound;
    if( entry->globalHead == NULL )
	return NotFound;

    //*content = entry->globalHead;
    for( tmp = entry->globalHead; tmp != NULL ; tmp = tmp->next ){
	if( strcmp( sourceDir , tmp->itsDir ) == 0 ){
	    if( strcmp( source, tmp->source ) == 0 ){
		if( tmp->line == line ){
		    *content = tmp;
		    retval = Itself;
		    break;
		}else{
		    if( in_element ) {
			if(element_type != NULL && is_match( tmp->type, element_type )){
			    retval=Found;
			    *content = tmp;
			    maxNum = INT_MAX;
			}
		    }
		    if( !in_element ){ 
			if(!(element_type != NULL && is_match( tmp->type, element_type ))){
			    retval=Found;
			    *content = tmp;
			    maxNum = INT_MAX;
			}
		    }

		}
	    }else if( maxNum != INT_MAX ) {
		 if( in_element ) {
			if(element_type != NULL && is_match( tmp->type, element_type )){
			    retval=Found;
			    *content = tmp;
			    maxNum = INT_MAX;
			}
		    }
		    if( !in_element ){ 
			if(!(element_type != NULL && is_match( tmp->type, element_type ))){
			    retval=Found;
			    *content = tmp;
			    maxNum = INT_MAX;
			}
		    }
	    }
	}
	else{
	    if( in_element ) {
		if(element_type != NULL && is_match( tmp->type, element_type )){
		    retval=Found;
		    common_dir = get_common_dir( sourceDir, tmp->itsDir );
		    nowNum = strlen( common_dir );
		    eFree( common_dir );
		    if( nowNum > maxNum ){
			maxNum = nowNum;
			*content = tmp;
		    }
		}
	    }
	    if( !in_element ){ 
		if(!(element_type != NULL && is_match( tmp->type, element_type ))){
		    retval=Found;
		    common_dir = get_common_dir( sourceDir, tmp->itsDir );
		    nowNum = strlen( common_dir );
		    eFree( common_dir );
		    if( nowNum > maxNum ){
			maxNum = nowNum;
			*content = tmp;
		    }
		}
	    }
	}
    }
    eFree( sourceDir);
    return retval;
}


/*
   켱 Ͽ  Ѵ.
   켱 
   0. static  itself  . 
   1.  source  static  
   2. global itself  .
   3. ٸ source global   丮  
   4. ٸ source global ̰ ٸ 丮

   content   ּҸ ִ´.

   return value 0: not found	1: found but, not itself
		2: found and itself
   
   in_element ũ Ȯ ڴ ּ ǵμ, 
   c ü 
   str{
   	int a;
	int b;
   }
    str   ִ struct_type ̰,
   a  b  element_type ȴ. 
    struct_type element_type  ̻   Ƿ ó Ȯ ,
       Ű    ְ ȴ. 

 */
extern int lookupKeyword (const char *const keyword, int line,
				const char *const source, lang_type language,
				const entryContent * *content, boolean in_element)
{
    const unsigned long hashedValue = hashValue (keyword);
    hashEntry *entry = getHashTableEntry (hashedValue);
    int static_retval=NotFound, global_retval=NotFound, retval=NotFound;
    entryContent    *static_cont, *global_cont;

    while (entry != NULL)
    {
	if (language == entry->language  &&  strcmp (keyword, entry->keyword) == 0)
	{
	    static_retval  = getStaticContent( entry, keyword, line, source, 
		    (const entryContent**)&static_cont, pl_table[language].element_type , in_element);
	    if( static_retval != Itself )
	    {
		global_retval	= getGlobalContent( entry,keyword, line, source,
			(const entryContent**) &global_cont, pl_table[language].element_type, in_element);
		if( global_retval == Itself ){
		    *content = global_cont;
		    retval = global_retval;
		    break;
		} else if( static_retval == NotFound && global_retval == Found ){
		    *content = global_cont;
		    retval = global_retval;
		    break;
		}
	    }
	    retval = static_retval;
	    *content = static_cont;
	    break;
	}
	entry = entry->next;
    }
    return retval;
}

static void freeEntryContent ( hashEntry *entry )
{
    entryContent *tmp,*next;
    tmp = entry->globalHead;
    while( tmp != NULL )
    {
	next = tmp->next;
	eFree( tmp );
	tmp = next;
    }

    tmp = entry->staticHead;
    while( tmp != NULL )
    {
	next = tmp->next;
	eFree( tmp );
	tmp = next;
    }
    return;
}

extern void freeKeywordTable (void)
{
    if (HashTable != NULL)
    {
	unsigned int i;

	for (i = 0  ;  i < TableSize  ;  ++i)
	{
	    hashEntry *entry = HashTable [i];

	    while (entry != NULL)
	    {
		hashEntry *next = entry->next;
		freeEntryContent( entry );
		eFree (entry);
		entry = next;
	    }
	}
	eFree (HashTable);
    }
}
/*
   add_keyword ϱ ռ ݵ غƾ ϴ Լ 
   ̹ global tag ϵǾ ִٸ, static ̰ Ѵ
   ⼭ false  static ȴ.
*/
extern boolean is_global (const char *const keyword, const char *const source,
			lang_type language,char type, int line )
{
    const unsigned long hashedValue = hashValue (keyword);
    hashEntry *tableEntry = getHashTableEntry (hashedValue);
    hashEntry *entry = tableEntry;
    const entryContent *dummy;

    hashEntry *prev = NULL;
    while (entry != NULL)
    {
	if (language == entry->language  &&
	    strcmp (keyword, entry->keyword) == 0)
	{
	    if( getGlobalContent( entry, NULL, line, source,(const entryContent**) &dummy, NULL ,FALSE) == Itself)
		return TRUE;
	    else
		return FALSE;
	}
	prev = entry;
	entry = entry->next;
    }
    return FALSE;
}



#ifdef DEBUG

static void printContent (const hashEntry *const entry , scope s)
{
    entryContent *tmp;
    if( s == Global ){
	printf("\t\tGlobal\n");
	tmp = entry->globalHead;
    }
    else{
	printf("\t\tStatic\n");
	tmp = entry->staticHead;
    }

    for( ; tmp != NULL ; tmp = tmp->next ){
	printf("\t\t[%s] [%d] [%d]\n", tmp->source, tmp->line, tmp->value);
    }

}
static void printEntry (const hashEntry *const entry)
{
    printf ("  %-15s %d\n", entry->keyword, entry->language);
    printContent( entry, Static);
    printContent( entry, Global);
}


static unsigned int printBucket (const unsigned int i)
{
    hashEntry **const table = getHashTable ();
    hashEntry *entry = table [i];
    unsigned int measure = 1;
    boolean first = TRUE;

    printf ("%2d:", i);
    if (entry == NULL)
	printf ("\n");
    else while (entry != NULL)
    {
	if (! first)
	    printf ("    ");
	else
	{
	    printf (" ");
	    first = FALSE;
	}
	printEntry (entry);
	entry = entry->next;
	measure = 2 * measure;
    }
    return measure - 1;
}

extern void printKeywordTable (void)
{
    unsigned long emptyBucketCount = 0;
    unsigned long measure = 0;
    unsigned int i;

    for (i = 0  ;  i < TableSize  ;  ++i)
    {
	const unsigned int pass = printBucket (i);

	measure += pass;
	if (pass == 0)
	    ++emptyBucketCount;
    }

    printf ("spread measure = %ld\n", measure);
    printf ("%ld empty buckets\n", emptyBucketCount);
}

#endif

/* vi:set tabstop=8 shiftwidth=4: */
